import { NextRequest, NextResponse } from 'next/server';
import { requireApiAuth } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { productUpdateSchema } from '@/lib/validation';

type Context = {
  params: {
    id: string;
  };
};

export async function PUT(request: NextRequest, context: Context) {
  const authError = await requireApiAuth(request);
  if (authError) return authError;

  let body: unknown;
  try {
    body = await request.json();
  } catch {
    return NextResponse.json({ error: 'JSON inválido' }, { status: 400 });
  }

  const parsed = productUpdateSchema.safeParse(body);
  if (!parsed.success) {
    return NextResponse.json({ error: parsed.error.issues[0]?.message || 'Datos inválidos' }, { status: 400 });
  }

  try {
    const product = await prisma.product.update({
      where: { id: context.params.id },
      data: {
        ...parsed.data,
        sku: parsed.data.sku === undefined ? undefined : parsed.data.sku || null,
        barcode: parsed.data.barcode === undefined ? undefined : parsed.data.barcode || null,
        supplier: parsed.data.supplier === undefined ? undefined : parsed.data.supplier || null
      }
    });

    return NextResponse.json(product);
  } catch {
    return NextResponse.json({ error: 'No se pudo actualizar el producto' }, { status: 400 });
  }
}

export async function DELETE(request: NextRequest, context: Context) {
  const authError = await requireApiAuth(request);
  if (authError) return authError;

  try {
    await prisma.product.update({
      where: { id: context.params.id },
      data: { isActive: false }
    });

    return NextResponse.json({ ok: true });
  } catch {
    return NextResponse.json({ error: 'No se pudo eliminar el producto' }, { status: 400 });
  }
}
